//=============================================================================
//  Keke_AnytimeFontChange - いつでもフォント変更
// バージョン: 1.0
//=============================================================================
// Copyright (c) 2020 ケケー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ゲーム中にパッとフォントを変更する
 * @author ケケー
 * 
 * @help
 * ゲーム中、いつでもフォントを変更できるよ
 * あらかじめフォントのファイル名と呼び出すための名前を登録しておき、
 * あとは名前を書くだけでパッと変えることができる
 *
 *
 *
 * ◉ ファイル投入 ◉
 * まずはフォントファイルを投入する
 * 好きなファイルをfontsフォルダの中に入れればよい
 *
 *
 * ◉ フォント登録 ◉
 * 次にプラグインパラメータでフォントを登録する
 * 入力事項は
 *
 * ■呼び出し名
 * ■ファイル名
 *
 * ▼ 呼び出し名
 * フォントを呼び出す時に書く名前だ
 * どんな名前でもよい。覚えやすく書きやすい名前にしよう
 *
 * ▼ ファイル名
 * 実際のファイルの名前
 * fontsフォルダに入れたファイルの名前をそのままコピーしてくればよい
 *
 * なお、フォント登録はリスト形式になっており、
 * 空欄ダブルクリックで何個でも登録できる
 *
 *
 * ◉ フォント変更 ◉
 * それでは実際にフォントを変更してみよう
 * 変更のしかたは二種類ある
 *
 * ■プラグインコマンド
 * ■制御文字
 *
 * ▼ プラグインコマンドの場合
 *
 * プラグインコマンド → Keke_AnyTimeFontChange → 呼び出し名
 *
 * ここにパラメータで登録した呼び出し名を書く
 *
 * ▼ 制御文字の場合
 * 
 * メッセージ中にこれを書く。書く場所はどこでもよい
 * \呼び出し名\fn
 *
 * たとえば呼び出し名が kee なら
 * \kee\fn
 *
 * fn は fontName の略だ
 *
 * 呼び出し名は日本語でも数字でもよい
 * \ケー\fn
 * \1\fn
 *
 * こう書くとメインフォントに戻す
 * \m\fn
 *
 * m は mainFont の略だ
 * このように m はメインフォント呼び出しに使っているので、
 * 呼び出し名には使わない方がよい
 *
 *
 * ◉ 適用範囲 ◉
 * フォント変更の適用範囲はメッセージだけでなく、
 * メニュー画面やショップ画面など、文字を使うすべてに及ぶ
 * そのため色々な使い道が考えられる
 *
 * ・いかがわしいショップではフォントも怪しい感じにしてみよう
 *
 * ・おどろおどろしい場面ではメニューのフォントもおどろおどろしくしてみよう
 *
 * など、想像力しだいで色々と面白い演出ができるハズだ
 * フォントは面白い
 *
 *
 *
 * ◉ 利用規約 ◉
 * MITライセンスのもと、好きに使ってくれて大丈夫
 * ただし作者は何も責任を負わないよ
 * また著作権は『ケケー』にあるよ
 *
 *
 *
 * @param フォント登録リスト
 * @desc 使うフォントのリスト。呼び出し名は自由に。ファイル名は拡張子まで入れること。空欄ダブルクリックで何個でも追加できる
 * @type struct<fontCfg>[]
 * @default []
 *
 *
 *
 * @command fontChange
 * @text フォント変更
 * @desc フォントを変更する。パラメータで登録したフォント名を入力することでそのフォントに変更できる
 * @arg fontName
 * @type string
 * @text 呼び出し名
 * @desc フォントの呼び出し名。パラメータで登録した名前を入力する
 *
 * @command fontReturn
 * @text フォント戻す
 * @desc フォントをメインフォントに戻す
 */

/*~struct~fontCfg:
 * @param 呼び出し名
 * @desc フォントの名前。フォントを呼び出すのに使う。何でもよい。覚えやすく書きやすい名前にしよう
 * @param ファイル名
 * @desc フォントファイルの名前。fontsフォルダに置いてあるファイルの名前をそのまま入力する。拡張子まで入れること
 */

(() => {
    // プラグイン名
    const pluginName = "Keke_AnyTimeFontChange";
    
    
    
    //--  文字列オート変換 /ベーシック  --//
    
    // 文字列のハッシュ化
    String.prototype.toHashKe = function() {
        let hash = {};
        let str = this;
        if (!str.length) { return; }
        const strs = JSON.parse(str);
        let val = null;
        let val2 = null;
        for (let key in strs) {
            val = strs[key];
            if (!key || !val) { continue; }
            val2 = val.toAutoKe(key);
            hash[key] = val2;
        }
        return hash;
    };
    
    // 文字列のリスト化
    String.prototype.toListKe = function() {
        let array = JSON.parse(this);
        return array.map((val, i) => {
            return val.toAutoKe();
        }, this);
    };
    
    // 文字列の自動処理
    String.prototype.toAutoKe = function(key = "") {
        let val = this;
        let val2 = null;
        let match = null;
        let end = false;
        if (!end) {
            if (val[0] == "{") {
                val2 = val.toHashKe();
                end = true;
            }
        }
        if (!end) {
            if (val[0] == "[") {
                val2 = val.toListKe();
                end = true;
            }
        }
        if (!end) { val = val + ","; }
        if (!end) {
            match = val.match(/^\s*(-?\d+,\s*-?\d+,\s*-?\d+,?\s*-?\d*\.?\d*)\s*,/);
            if (match && !val.match(/[a-z]/)) {
                if (key.includes("カラー") && !key.includes("トーン") && !key.includes("チェンジ") &&  !key.includes("選択")) {
                    val2 =  "rgba(" +  match[1] + ")";
                } else {
                    val2 =  eval("[" +  match[1] + "]");
                }
                end = true;
            }
        }
        if (!end) {
            match = val.match(/(-?\d+\.?\d*),\s*/g);
            if (match && match.length >= 2 && !val.match(/[a-z]/)) {
                val2 =  eval("[" + match.reduce((r, s) => r + s) + "]");
                end = true;
            }
        }
        if (!end) {
            match = val.match(/^(true|false)\s*,/);
            if (match) {
                val2 =  match[1] == "true" ? true : false;
                end = true;
            }
        }
        if (!end) {
            match = val.match(/^(-?\d+\.?\d*)\s*,/);
            if (match && !val.match(/[a-z]/)) {
                val2 = Number(match[1]); end = true;
                end = true;
            }
        }
        if (!end) {
            if (val[0] == "\"") { val = val.slice(1); }
            val2 = val.slice(0, -1);
        }
        return val2;
    };
    
    
    
    //--  プラグインコマンド基本 /ベーシック  --//
    
    // プラグインコマンド呼び出しプリターを保存
    const _PluginManager_callCommand = PluginManager.callCommand;
    PluginManager.callCommand = function(self, pluginName, commandName, args) {
        $gameTemp._pluginCmdPreterKe = self;
        _PluginManager_callCommand.call(this, self, pluginName, commandName, args);
    };
    



    //--  パラメータ受け取り  --//
    
    var parameters = PluginManager.parameters(pluginName);
    var keke_fontList = parameters["フォント登録リスト"].toListKe()



    //--  プラグインコマンド  --//
    
    // フォント変更
    PluginManager.registerCommand(pluginName, "fontChange", args => {
        // イニット
        const fontName = args.fontName;
        // ファイル名を取得
        const fileName = $gameTemp.getFontFileNameKe(fontName);
        // あったらフォント変更
        if (fileName) {
            FontManager._states["rmmz-mainfont"] = null;
            FontManager.load("rmmz-mainfont", fileName);
        }
        // フォントウェイト開始
        $gameTemp._pluginCmdPreterKe._waitsFontKe = 1;
    });
    
    // フォント戻す
    PluginManager.registerCommand(pluginName, "fontReturn", args => {
        // メインフォントに変更
        FontManager._states["rmmz-mainfont"] = null;
        FontManager.load("rmmz-mainfont", $dataSystem.advanced.mainFontFilename);
        // フォントウェイト開始
        $gameTemp._pluginCmdPreterKe._waitsFontKe = 1;
    });
    
    // フォントウェイト(インタープリター)
    const _Game_Interpreter_updateWait = Game_Interpreter.prototype.updateWait;
    Game_Interpreter.prototype.updateWait = function() {
        let result = _Game_Interpreter_updateWait.call(this);
        // ウェイト中なら
        if (this._waitsFontKe) {
            // フォントのロード完了を待つ
            if (FontManager._states["rmmz-mainfont"] != "loaded") {
                result = true;
            // 完了したらウェイト削除
            } else {
                this._waitsFontKe = null;
            }
        }
        return result 
    };
    
    


     
    //--  共通処理  --//
    
    // フォントファイル名の取得
    Game_Temp.prototype.getFontFileNameKe = function(fontName) {
        // フォント名がないならリターン
        if (!fontName) { return ""; }
         // mならメインフォントを取得
        if (fontName == "m") {
            return $dataSystem.advanced.mainFontFilename;
        }
        // イニット
        let fileName = "";
        // フォントリスト展開
        for (const cfg of keke_fontList) {
            // 同じ名前があったらファイル名を取得
            if (cfg["呼び出し名"] == fontName) {
                fileName = cfg["ファイル名"];
            }
        }
        return fileName;
    };
    
    
    
    
    
    //--  制御文字でのフォント変更  --//
    
    // フォント変更があったら変更してウェイト
    const _Window_Message_prototype_startMessage = Window_Message.prototype.startMessage;
    Window_Message.prototype.startMessage = function() {
        // 非ウェイト中なら
        if (this._waitsFontKe == null) {
            // テキスト取得
            const text = $gameMessage._texts.join("/n");
            // 制御文字マッチしたら
            const match = text.match(/\\([^\\]+)\\fn/);
            if (match) {
                // フォント名を取得
                const fontName = match[1];
                // ファイル名を取得
                const fileName = $gameTemp.getFontFileNameKe(fontName);
                // フォント変更
                FontManager._states["rmmz-mainfont"] = null;
                FontManager.load("rmmz-mainfont", fileName);
                // フォントウェイト開始
                this._waitsFontKe = 1;
                return;
            }
        }
        // フォントウェイト消去
        this._waitsFontKe  = null;
        _Window_Message_prototype_startMessage.call(this);
    };
    
    
    // フォントウェイト(メッセージ)
    const _Window_Message_updateWait = Window_Message.prototype.updateWait;
    Window_Message.prototype.updateWait = function() {
        let result =  _Window_Message_updateWait.call(this);
        // ウェイト中なら
        if (this._waitsFontKe) {
            // フォントのロード完了を待つ
            if (FontManager._states["rmmz-mainfont"] != "loaded") {
                result = true;
            // 完了したらウェイトを0に
            } else {
                this._waitsFontKe = 0;
            }
        }
        return result;
    };
    
    
    // フォント変更文字を消去
    const _Game_Message_allText = Game_Message.prototype.allText;
    Game_Message.prototype.allText = function() {
        let text = _Game_Message_allText.call(this);
        if (text) {
            text = text.replace(/\\([^\\]+)\\fn/g, "");
        }
        return text
    };
        
})();